/**
    \file DeadReckoning_3x3.cpp
    Implementation of DeadReckoning_3x3 distance transform class which, given
    and input binary image, calculates the corresponding distance transform.

    \author George J. Grevera, Ph.D., ggrevera@sju.edu

    Copyright (C) 2002, George J. Grevera

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
    USA or from http://www.gnu.org/licenses/gpl.txt.

    This General Public License does not permit incorporating this
    code into proprietary programs.  (So a hypothetical company such
    as GH (Generally Hectic) should NOT incorporate this code into
    their proprietary programs.)
 */
/*
     no OUT OF BOUNDS checks!
*/
#include <stdlib.h>
#include "DeadReckoning_3x3.h"

using namespace std;
//----------------------------------------------------------------------
/*
Input : (C,I) - a 2D binary scene of domain size X by Y
Output: (C,d) - a 2D grey scene of domain size X and Y representing the
                distance scene
*/
void DeadReckoning_3x3::doTransform ( const unsigned char* const I ) {
    const double dx=1.0;
    const double dy=1.0;
    borderCheck(I);
    cleanUp();
    int x, y;

    //initialize d
    double* d = (double*)malloc( ySize*xSize*sizeof(double)   );
    assert( d!=NULL );
    this->p = (P*)malloc( ySize*xSize*sizeof(P) );
    assert( this->p!=NULL );
    for (y=0; y<ySize; y++) {
        for (x=0; x<xSize; x++) {
            const int i=sub(x,y);
            d[i] = DistanceTransform::FloatInfinity;
            //pointer to immediate interior or exterior i.e.
            //p(x,y).x is the x coordinate of the closest boundary
            //point (and similarly for p(x,y).y)
            this->p[i].x = this->p[i].y = -1;
        }
    }

    //initialize immediate interior & exterior elements
    //double halfDx = 0, halfDy = 0;
    for (y=1; y<ySize-1; y++) {
        for (x=1; x<xSize-1; x++) {
            if ( I[sub(x-1,y)] != I[sub(x,y)] || I[sub(x+1,y)] != I[sub(x,y)] ||
                 I[sub(x,y-1)] != I[sub(x,y)] || I[sub(x,y+1)] != I[sub(x,y)] ) 
{
                    const int i=sub(x,y);
                    d[i] = 0;
                    this->p[i].x = x;
                    this->p[i].y = y;
            }
        }
    }

    const double dxy = sqrt(2.0);  //diagonal distance

        #define _check(X,Y,Delta)                             \
            i1=sub((X),(Y));                                  \
            if (d[i1]+(Delta) < d[i2]) {                      \
                this->p[i2] = this->p[i1];                    \
                const double deltaX = (p[i1].x - x);          \
                const double deltaY = (p[i1].y - y);          \
                d[i2] = sqrt(deltaX*deltaX + deltaY*deltaY);  \
            }

        //perform the first pass
        for (y=1; y<ySize-1; y++) {
            for (x=1; x<xSize-1; x++) {
                int i1;
                const int i2=sub(x,y);

                _check( x-1, y-1, dxy );
                _check( x,   y-1, dy  );
                _check( x+1, y-1, dxy );
                _check( x-1, y,   dx  );
#if 0
                //extra:
                _check( x+1, y,   dx  );
                _check( x-1, y+1, dxy );
                _check( x,   y+1, dy  );
                _check( x+1, y+1, dxy );
#endif
            }
        }

        //perform the final pass
        for (y=ySize-2; y>=1; y--) {
            for (x=xSize-2; x>=1; x--) {
                int i1;
                const int i2=sub(x,y);

                _check( x+1, y,   dx  );
                _check( x-1, y+1, dxy );
                _check( x,   y+1, dy  );
                _check( x+1, y+1, dxy );
#if 0
                //extra:
                _check( x-1, y-1, dxy );
                _check( x,   y-1, dy  );
                _check( x+1, y-1, dxy );
                _check( x-1, y,   dx  );
#endif
            }
        }

    finish(I, d);
}
//----------------------------------------------------------------------
void DeadReckoning_3x3::doTransform ( const unsigned char* const I,
    const double dx, const double dy, const bool halfIsZero )
{
    borderCheck(I);
    cleanUp();
    int x, y;

    //initialize d
    double* d = (double*)malloc( ySize*xSize*sizeof(double)   );
    assert( d!=NULL );
    this->p = (P*)malloc( ySize*xSize*sizeof(P) );
    assert( this->p!=NULL );
    for (y=0; y<ySize; y++) {
        for (x=0; x<xSize; x++) {
            const int i=sub(x,y);
            d[i] = DistanceTransform::FloatInfinity;
            //pointer to immediate interior or exterior i.e.
            //p(x,y).x is the x coordinate of the closest boundary
            //point (and similarly for p(x,y).y)
            this->p[i].x = this->p[i].y = -1;
        }
    }
    //initialize immediate interior & exterior elements
    double halfDx = dx / 2, halfDy = dy / 2;
    if (halfIsZero)  halfDx=halfDy=0;
    for (y=1; y<ySize-1; y++) {
        for (x=1; x<xSize-1; x++) {
            if (I[sub(x-1,y)] != I[sub(x,y)] || I[sub(x+1,y)] != I[sub(x,y)]) {
                if (halfDx < d[sub(x,y)]) {
                    const int i=sub(x,y);
                    d[i] = halfDx;
                    this->p[i].x = x;
                    this->p[i].y = y;
                }
            }
            if (I[sub(x,y-1)] != I[sub(x,y)] || I[sub(x,y+1)] != I[sub(x,y)]) {
                if (halfDy < d[sub(x,y)]) {
                    const int i=sub(x,y);
                    d[i] = halfDy;
                    this->p[i].x = x;
                    this->p[i].y = y;
                }
            }
        }
    }

    const double dxy = sqrt(dx*dx +  dy*dy);  //diagonal distance

    if (halfIsZero) {
        #undef  _check
        #define _check(X,Y,D)                                 \
            i1=sub((X),(Y));                                  \
            if (d[i1]+(D) < d[i2]) {                          \
                this->p[i2] = this->p[i1];                    \
                const double deltaX = (p[i1].x - x)*dx;       \
                const double deltaY = (p[i1].y - y)*dy;       \
                d[i2] = sqrt(deltaX*deltaX + deltaY*deltaY);  \
            }

        //perform the first pass
        for (y=1; y<ySize-1; y++) {
            for (x=1; x<xSize-1; x++) {
                int i1;
                const int i2=sub(x,y);

                _check( x-1, y-1, dxy );
                _check( x,   y-1, dy  );
                _check( x+1, y-1, dxy );
                _check( x-1, y,   dx  );
            }
        }

        //perform the final pass
        for (y=ySize-2; y>=1; y--) {
            for (x=xSize-2; x>=1; x--) {
                int i1;
                const int i2=sub(x,y);

                _check( x+1, y,   dx  );
                _check( x-1, y+1, dxy );
                _check( x,   y+1, dy  );
                _check( x+1, y+1, dxy );
            }
        }
    } else {
        #undef _check
        #define _check(X,Y,D)                                         \
            i1=sub((X),(Y));                                          \
            if (d[i1]+(D) < d[i2]) {                                  \
                this->p[i2] = this->p[i1];                            \
                const double deltaX = (p[i1].x - x)*dx;               \
                const double deltaY = (p[i1].y - y)*dy;               \
                const int i3 = sub(p[i1].x, p[i1].y);                 \
                d[i2] = sqrt(deltaX*deltaX + deltaY*deltaY) + d[i3];  \
            }

        //perform the first pass
        for (y=1; y<ySize-1; y++) {
            for (x=1; x<xSize-1; x++) {
                int i1;
                const int i2=sub(x,y);

                _check( x-1, y-1, dxy );
                _check( x,   y-1, dy  );
                _check( x+1, y-1, dxy );
                _check( x-1, y,   dx  );
            }
        }

        //perform the final pass
        for (y=ySize-2; y>=1; y--) {
            for (x=xSize-2; x>=1; x--) {
                int i1;
                const int i2=sub(x,y);

                _check( x+1, y,   dx  );
                _check( x-1, y+1, dxy );
                _check( x,   y+1, dy  );
                _check( x+1, y+1, dxy );
            }
        }
    }

    finish(I, d);
}
//----------------------------------------------------------------------

